(in-package :org.net-snmp)

(defclass snmp-session ()
  ((peername :reader snmp-peername
	     :initarg :peername
	     :type string
	     :initform "localhost")
   (version :reader snmp-version
	    :initarg :version
	    :type integer
	    :initform +snmp-version-2c+)
   (community :reader snmp-community
	      :initarg :community
	      :type string
	      :initform "public")
   c-session))

(defmethod shared-initialize :after ((instance snmp-session) slot-names &rest initargs)
  (declare (ignore slot-names initargs))
  (with-slots (peername version community c-session) instance
    (progn
      (setf c-session (foreign-alloc 'c-snmp-session))
      (c-snmp-session-init c-session)
      (with-foreign-slots ((c-peername c-version c-community c-community-len)
			   c-session c-snmp-session)
	(setf c-peername (foreign-string-alloc peername)
	      c-version version
	      c-community (foreign-string-alloc community)
	      c-community-len (length community))))))

(defclass oid ()
  ((name :type string :reader oid-name)
   c-oids
   c-oid-len))

(defmethod oid-length ((instance oid))
  (when (slot-boundp instance 'c-oid-len)
    (mem-ref (slot-value instance 'c-oid-len) 'c-size-type)))

(defmethod make-oid ((oid-string string))
  (let ((oid (make-instance 'oid)))
    (with-slots (name c-oids c-oid-len) oid
      (progn
	(setf name oid-string
	      c-oids (foreign-alloc 'c-oid :count +max-oid-len+)
	      c-oid-len (foreign-alloc 'c-size-type :initial-element +max-oid-len+))
	(if (eq (elt oid-string 0) #\.)
	    (c-read-objid oid-string c-oids c-oid-len)
	    (c-get-node oid-string c-oids c-oid-len))
	oid))))

(defmethod snmp-msg-get ((s snmp-session) (o oid))
  (let ((ss (c-snmp-open (slot-value s 'c-session)))
	(pdu (c-snmp-pdu-create +snmp-msg-get+))
	(response (foreign-alloc :pointer :initial-element (null-pointer))))
    (progn
      (c-snmp-add-null-var pdu
			   (slot-value o 'c-oids)
			   (mem-ref (slot-value o 'c-oid-len) 'c-size-type))
      (let ((status (c-snmp-synch-response ss pdu response)))
	(if (and (= status +snmp-stat-success+)
		 (= (foreign-slot-value (mem-aref response :pointer) 'c-snmp-pdu 'c-errstat)
		    +snmp-err-noerror+))
	    (let ((vars (foreign-slot-value (mem-aref response :pointer)
					    'c-snmp-pdu 'c-variables)))
	      (c-print-variable (foreign-slot-value vars 'c-variable-list 'c-name)
				(foreign-slot-value vars 'c-variable-list 'c-name-length)
				vars))
	    nil))
      (c-snmp-pdu-free (mem-aref response :pointer))
      (c-snmp-close ss)
      nil)))

(defmethod snmp-msg-get ((s snmp-session) (o string))
  (snmp-msg-get s (make-oid o)))

(defmethod snmp-msg-get ((s string) (o string))
  (snmp-msg-get (make-instance 'snmp-session :peername s) o))

