(in-package :org.net-snmp)

(defctype c-size-type :long)

(defcstruct c-snmp-pdu
  "The snmp protocol data unit."
  ;;
  ;; Protocol-version independent fields
  ;;
  ;; snmp version
  (c-version :long)
  ;; Type of this PDU
  (c-command :int)
  ;; Request id - note: not incremented on retries
  (c-reqid :long)
  ;; Message id for V3 messages note: incremented for each retry
  (c-msgid :long)
  ;; Unique ID for incoming transactions
  (c-transid :long)
  ;; Session id for AgentX messages
  (c-sessid :long)
  ;; Error status (non_repeaters in GetBulk)
  (c-errstat :long)
  ;; Error index (max_repetitions in GetBulk)
  (c-errindex :long)
  ;; Uptime
  (c-time :ulong)
  (c-flags :ulong)

  (c-security-model :int)
  ;; noAuthNoPriv, authNoPriv, authPriv
  (c-security-level :int)
  (c-msg-parse-model :int)
  
  ;;
  ;; Transport-specific opaque data.  This replaces the IP-centric address
  ;; field.
  ;;

  (c-transport-data :pointer)
  (c-transport-data-length :int)

  ;;
  ;; The actual transport domain.  This SHOULD NOT BE FREE()D.
  ;;

  (c-t-domain :pointer)
  (c-t-domain-len c-size-type)

  (c-variables :pointer)

  ;;
  ;; SNMPv1 & SNMPv2c fields
  ;;
  ;; community for outgoing requests.
  (c-community :pointer)
  ;; length of community name.
  (c-community-len c-size-type)

  ;;
  ;; Trap information
  ;;
  ;; System OID
  (c-enterprise :pointer)
  (c-enterprise-length c-size-type)
  ;; trap type
  (c-trap-type :long)
  ;; specific type
  (c-specific-type :long)
  ;; This is ONLY used for v1 TRAPs
  (c-agent-addr :pointer)

  ;;
  ;; SNMPv3 fields
  ;;
  ;; context snmpEngineID
  (c-context-engine-id :pointer)
  ;; Length of contextEngineID
  (c-context-engine-id-len c-size-type)
  ;; authoritative contextName
  (c-context-name :string)
  ;; Length of contextName
  (c-context-name-len c-size-type)
  ;; authoritative snmpEngineID for security
  (c-security-engine-id :pointer)
  ;; Length of securityEngineID
  (c-security-engine-id-len c-size-type)
  ;; on behalf of this principal
  (c-security-name :string)
  ;; Length of securityName.
  (c-security-name-len c-size-type)
    
  ;;
  ;; AgentX fields
  ;; (also uses SNMPv1 community field)
  ;;
  (c-priority :int)
  (c-range-subid :int)

  (c-security-state-ref :pointer))

(defctype c-netsnmp-pdu c-snmp-pdu)
(defctype c-netsnmp-callback :pointer)
(defctype c-authenticator :pointer)

(defmacro def-snmp-session ()
  `(defcstruct c-snmp-session
     "The snmp session structure."
     ;;
     ;; Protocol-version independent fields
     ;;
     ;; snmp version
     (c-version :long)
     ;; Number of retries before timeout.
     (c-retries :int)
     ;; Number of uS until first timeout, then exponential backoff
     (c-timeout :long)
     (c-flags :ulong)
     (c-subsession :pointer)
     (c-next :pointer)
     ;; Domain name or dotted IP address of default peer
     (c-peername :string)
     ;; UDP port number of peer.
     (c-remote-port :ushort)
     ;; My Domain name or dotted IP address, 0 for default
     (c-localname :string)
     ;; My UDP port number, 0 for default, picked randomly
     (c-local-port :ushort)
     ;;
     ;; Authentication function or NULL if null authentication is used
     ;;
     (c-authenticator c-authenticator)
     ;; Function to interpret incoming data
     (c-callback c-netsnmp-callback)
     ;;
     ;; Pointer to data that the callback function may consider important
     ;;
     (c-callback-magic :pointer)
     ;; copy of system errno
     (c-system-errno :int)
     ;; copy of library errno
     (c-system-snmp-errno :int)
     ;; Session id - AgentX only
     (c-sessid :long)
     ;;
     ;; SNMPv1 & SNMPv2c fields
     ;;
     ;; community for outgoing requests.
     (c-community :pointer)
     ;; Length of community name.
     (c-community-len c-size-type)
     ;; Largest message to try to receive.
     (c-rcv-msg-max-size c-size-type)
     ;; Largest message to try to send.
     (c-snd-msg-max-size c-size-type)
     ;;
     ;; SNMPv3 fields
     ;;
     ;; are we the authoritative engine?
     (c-is-authoritative :uchar)
     ;; authoritative snmpEngineID
     (c-context-engine-id :pointer)
     ;; Length of contextEngineID
     (c-context-engine-id-len c-size-type)
     ;; initial engineBoots for remote engine
     (c-engine-boots :uint)
     ;; initial engineTime for remote engine
     (c-engine-time :uint)
     ;; authoritative contextName
     (c-context-name :string)
     ;; Length of contextName
     (c-context-name-len c-size-type)
     ;; authoritative snmpEngineID
     (c-security-engine-id :pointer)
     ;; Length of contextEngineID
     (c-security-engine-id-len c-size-type)
     ;; on behalf of this principal
     (c-security-name :string)
     ;; Length of securityName.
     (c-security-name-len c-size-type)
     ;; auth protocol oid
     (c-security-auth-proto :pointer)
     ;; Length of auth protocol oid
     (c-security-auth-proto-len c-size-type)
     ;; Ku for auth protocol XXX
     (c-security-auth-key :uchar :count ,+usm-auth-ku-len+)
     ;; Length of Ku for auth protocol
     (c-security-auth-key-len c-size-type)
     ;; Kul for auth protocol
     (c-security-auth-local-key :pointer)
     ;; Length of Kul for auth protocol XXX
     (c-security-auth-local-key-len c-size-type)
     ;; priv protocol oid
     (c-security-priv-proto :pointer)
     ;; Length of priv protocol oid
     (c-security-priv-proto-len c-size-type)
     ;; Ku for privacy protocol XXX
     (c-security-priv-key :uchar :count ,+usm-priv-ku-len+)
     ;; Length of Ku for priv protocol
     (c-security-priv-key-len c-size-type)
     ;; Kul for priv protocol
     (c-security-priv-local-key :pointer)
     ;; Length of Kul for priv protocol XXX
     (c-security-priv-local-key-len c-size-type)
     ;; snmp security model, v1, v2c, usm
     (c-security-model :int)
     ;; noAuthNoPriv, authNoPriv, authPriv
     (c-security-level :int)
     ;; target param name
     (c-param-name :string)
     ;;
     ;; security module specific
     ;;
     (c-security-info :pointer)
     ;;
     ;; use as you want data
     ;;
     (c-myvoid :pointer)))

(defctype c-oid :ulong)

(defctype c-netsnmp-vardata :pointer)

(defmacro def-variable-list ()
  `(defcstruct c-variable-list
     ;; NULL for last variable
     (c-next-variable :pointer)
     ;; Object identifier of variable
     (c-name :pointer)
     ;; number of subid's in name
     (c-name-length c-size-type)
     ;; ASN type of variable
     (c-type :uchar)
     ;; value of variable
     (c-val c-netsnmp-vardata)
     ;; the length of the value to be copied into buf
     (c-val-len c-size-type)
     ;; 90 percentile < 24.
     (c-name-loc c-oid :count ,+max-oid-len+)
     ;; 90 percentile < 40.
     (c-buf :uchar :count 40)
     ;; (Opaque) hook for additional data
     (c-data :pointer)
     ;; callback to free above
     (c-data-free-hook :pointer)
     (c-index :int)))

(eval-when (:compile-toplevel :load-toplevel :execute)
  (def-snmp-session)
  (def-variable-list))

(defctype c-netsnmp-session c-snmp-session)
