
; see http://babbage.cs.qc.edu/IEEE-754/


(in-package :binary-io-utils)



(defun read-u2 (in)
  (let ((u2 0))
    (setf (ldb (byte 8 8) u2) (read-byte in))
    (setf (ldb (byte 8 0) u2) (read-byte in))
    u2))

(defun read-u4-littleendian (in)
  (let ((u4 0))
    (setf (ldb (byte 8 24) u4) (read-byte in))
    (setf (ldb (byte 8 16) u4) (read-byte in))
    (setf (ldb (byte 8 8) u4) (read-byte in))
    (setf (ldb (byte 8 0) u4) (read-byte in))
    u4))

(defun read-u4-bigendian (in)
  (let ((u4 0))
    (setf (ldb (byte 8 0) u4) (read-byte in))
    (setf (ldb (byte 8 8) u4) (read-byte in))
    (setf (ldb (byte 8 16) u4) (read-byte in))
    (setf (ldb (byte 8 24) u4) (read-byte in))
    u4))


(defun write-u2 (out value)
  (write-byte (ldb (byte 8 8) value) out)
  (write-byte (ldb (byte 8 0) value) out))



(defun write-u4-littleendian (out value)
  (write-byte (ldb (byte 8 24) value) out)
  (write-byte (ldb (byte 8 16) value) out)
  (write-byte (ldb (byte 8 8) value) out)
  (write-byte (ldb (byte 8 0) value) out))
  


(defun write-u4-bigendian (out value)
  (write-byte (ldb (byte 8 0) value) out)
  (write-byte (ldb (byte 8 8) value) out)
  (write-byte (ldb (byte 8 16) value) out)
  (write-byte (ldb (byte 8 24) value) out))


(defmacro gen-ieee-encoding (name type exponent-bits mantissa-bits)
  `(progn
				
    (defun ,(intern (format nil "~A-TO-IEEE-754"  name))  (float)
	(multiple-value-bind (significand expon sgn)
	    (integer-decode-float float)
	  (let* ((slen (integer-length significand))
		 (delta (- slen ,(1+ mantissa-bits)))
		 (sgn-norm (ash significand delta))
		 (ex (- (+ ,(+ mantissa-bits (1- (expt 2 (1- exponent-bits))) ) expon)
			delta))
		 (output (if (minusp sgn) (dpb 1 (byte  1 ,(+ mantissa-bits exponent-bits)) 0)
		       0))
	   (final (if (not (plusp ex))
		      (dpb (ldb (byte ,mantissa-bits 0) (ash sgn-norm   (1- ex)))  
			   (byte ,mantissa-bits 0)  output)
		      ;; or else .
		      (dpb (ldb (byte ,mantissa-bits 0) sgn-norm) (byte ,mantissa-bits 0)
			   (dpb ex (byte ,exponent-bits ,mantissa-bits) output)))))
	    final)))


    (defun ,(intern (format nil "IEEE-754-TO-~A" name))  (ieee) 
      (let* ((ex (ldb (byte ,exponent-bits ,mantissa-bits) ieee))
	     (sig (ldb (byte ,mantissa-bits 0) ieee))
	     (significand (if (zerop ex)
			      (ash sig 1) ;Hack! FIXME
			      (dpb 1 (byte 1 ,mantissa-bits) sig))) 
	     (ssigned (if (logbitp  ,(+ exponent-bits mantissa-bits) ieee)
			  (- significand)
			  significand))
	       
	     (aval 			 
	      (scale-float (coerce ssigned ,type)
			   (- ex
			      ,(+ (1- (expt 2 (1- exponent-bits)))
				  (1- mantissa-bits)
				  1
				  )  ))))
	aval))
))


(gen-ieee-encoding float-32 'single-float  8 23)
(gen-ieee-encoding float-64 'double-float 11 52)


(defun read-float32-bigendian (in)
  (ieee-754-to-float-32 (read-u4-bigendian in)))

(defun read-float32-littleendian (in)
  (ieee-754-to-float-32 (read-u4-littleendian in)))


(defun write-float32-bigendian (out value)
  (write-u4-bigendian out (float-32-to-ieee-754 value)))

(defun write-float32-littleendian (out value)
  (write-u4-littleendian out (float-32-to-ieee-754 value)))


